//==============================================================================
//
//	Copyright (c) 2016-
//	Authors:
//	* Joachim Klein <klein@tcs.inf.tu-dresden.de> (TU Dresden)
//
//------------------------------------------------------------------------------
//
//	This file is part of PRISM.
//
//	PRISM is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	PRISM is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with PRISM; if not, write to the Free Software Foundation,
//	Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//==============================================================================


#ifndef EXPORT_ITERATIONS_H
#define EXPORT_ITERATIONS_H

#include <cstdio>
#include <string>
#include <cmath>

#include <cudd.h>
#include <dd.h>
#include "dv.h"
#include "prism.h"
#include <odd.h>

class ExportIterations {
private:
	FILE *fp;
	std::string filename;

public:
	/**
	 * Constructor, provide filename.
	 */
	ExportIterations(const char* title = "", const char* filename = get_export_iterations_filename()) {
		fp = fopen(filename, "w");
		this->filename = filename;

		fprintf(fp, "<!DOCTYPE html>\n");
		fprintf(fp, "<html><head>\n");
		fprintf(fp, "<meta charset=\"utf-8\">\n");
		fprintf(fp, "<!-- HTML file automatically generated by the PRISM model checker -->\n");
		fprintf(fp, "<!-- For visualising the individual steps of a value iteration computation -->\n");
		fprintf(fp, "<!-- Loads supporting Javascript and CSS from www.prismmodelchecker.org -->\n");
		fprintf(fp, "<title>%s</title>\n", title);
		fprintf(fp, "<link rel='stylesheet' href='https://www.prismmodelchecker.org/js/res/iteration-vis-v1.css'>\n");
		fprintf(fp, "<script src=\"https://www.prismmodelchecker.org/js/res/d3.js-v4/d3.min.js\"></script>\n");
		fprintf(fp, "<body onload='init();'>\n");
		fprintf(fp, "<h1>%s</h1>\n", title);
		fprintf(fp, "<svg></svg>\n");
		fprintf(fp, "<script src=\"https://www.prismmodelchecker.org/js/res/iteration-vis-v1.js\"></script>\n");
	}

	/**
	 * Get the filename used for exporting.
	 */
	const std::string& getFileName() {
		return filename;
	}

	/**
	 * Export the given vector, with size n and given type (0 = normal, VI from below, 1 = VI from above)
	 */
	void exportVector(double *soln, int64_t n, int type) {
		fprintf(fp, "<script>addVector([");
		for (int64_t i = 0; i < n; i++) {
			if (i>0) fprintf(fp, ",");
			double d = soln[i];
			if (std::isinf(d)) {
				if (d > 0)
					fprintf(fp, "Infinity");
				else
					fprintf(fp, "-Infinity");
			} else {
				fprintf(fp, "%.17g", soln[i]);
			}
		}
		fprintf(fp, "],%d)</script>\n", type);
		fflush(fp);
	}

	/**
	 * Export the given MTBDD vector, with num_rvars row variables,
	 * odd for reachable state space and type (0= normal, VI from below, 1 = VI from above)
	 */
	void exportVector(DdNode *dd, DdNode **rvars, int num_rvars, ODDNode* odd, int type)
	{
		double* vec = mtbdd_to_double_vector(ddman, dd, rvars, num_rvars, odd);

		// get number of states
		int64_t n = odd->eoff + odd->toff;

		exportVector(vec, n, type);
		delete[] vec;
	}

	/** Destructor, close file */
	~ExportIterations() {
		fprintf(fp, "</body></html>\n");
		fclose(fp);
	}
};

#endif
